package main

import (
	"flag"
	"log"
	"net"
	"time"
)

type BlasterClient struct {
	conn *net.UDPConn
}

var blaster *BlasterClient
var LOG *log.Logger
var sec_ns int64

func init() {
	LOG = log.Default()
	_sec, _ := time.ParseDuration("1s")
	sec_ns = int64(_sec.Nanoseconds())
}

func main() {
	var arg_pkt_rate uint
	flag.UintVar(&arg_pkt_rate, "r", 0, "Starting packet send rate [optional]")
	flag.Parse()

	remote, err := net.ResolveUDPAddr("udp", "127.0.0.1:9825")
	if err != nil {
		panic("Failed to set up UDP client socket")
	}
	blaster = new(BlasterClient)
	blaster.conn, err = net.DialUDP("udp", nil, remote)

	udp_msg := []byte("ctr_X:1|c|#some:info,in:tags,and:other,random:garbage\n")

	var pps uint64 = 6500
	if arg_pkt_rate > 0 {
		pps = uint64(arg_pkt_rate)
	}

	var pps_f float32 = 0.0
	var mul float32 = 1.5
	// Ramp up the transmission speed
	for {
		LOG.Printf("Trying to blast with %d packets per second\n", pps)
		measured_rate := pump_pkts(pps, 20, udp_msg)
		LOG.Printf("Waiting for 2 seconds before next round...\n")
		time.Sleep(2 * time.Second)
		if measured_rate >= 25000.0 {
			mul = 1.25
		}
		pps_f = float32(pps) * mul
		pps = uint64(pps_f) // Rounds down
	}
}

func adapted_batch_size(pump_rate uint64) int64 {
	switch {
	case pump_rate > 200000:
		return 15000
	case pump_rate > 120000:
		return 10000
	case pump_rate > 75000:
		return 7500
	case pump_rate > 50000:
		return 4000
	default:
		return 1000
	}
}

func pump_pkts(pump_rate uint64, seconds uint64, msg []byte) float32 {
	packet_delay_ns := int64(float64(sec_ns) / float64(pump_rate))
	now := time.Now()
	end_at := now.Add(time.Second * time.Duration(seconds))
	sent := 0
	batch_size := adapted_batch_size(pump_rate)
	LOG.Printf("Batch size %d, average packet delay will be: %d ns\n", batch_size, packet_delay_ns)
	for {
		t := time.Now()
		if t.After(end_at) {
			break
		}

		batch := int32(batch_size)
		for batch >= 0 {
			blaster.conn.Write(msg)
			sent++
			batch--
		}
		// Average packet delay is set in nanoseconds. Account for batching.
		time.Sleep(time.Duration(packet_delay_ns*batch_size) * time.Nanosecond)
	}
	avg_rate := float32(sent) / float32(seconds)
	LOG.Printf("Sent %d packets (average rate %.01f pps)\n", sent, avg_rate)
	return avg_rate
}
