package main

import (
	"testing"
)

func TestTag(t *testing.T) {
	s := "tag:val"
	tags := ParseTags(s)
	if tags["tag"] != "val" {
		t.Fail()
	}
}

func TestTagsMany(t *testing.T) {
	s := "tag_str:yadda,tag_int:42,tag_float:4.25"
	tags := ParseTags(s)
	if tags["tag_str"] != "yadda" {
		t.Fail()
	}
	if tags["tag_int"] != "42" {
		t.Fail()
	}
	if tags["tag_float"] != "4.25" {
		t.Fail()
	}
}

func TestTagsManyTrailingComma(t *testing.T) {
	s := "tag_str:yadda,tag_int:42,tag_float:4.25,"
	tags := ParseTags(s)
	if tags["tag_str"] != "yadda" {
		t.Fail()
	}
	if tags["tag_int"] != "42" {
		t.Fail()
	}
	if tags["tag_float"] != "4.25" {
		t.Fail()
	}
}

func TestTagsManyTooManyCommas(t *testing.T) {
	s := ",tag_str:yadda,,,tag_int:42,tag_float:4.25,,,,"
	tags := ParseTags(s)
	if tags["tag_str"] != "yadda" {
		t.Fail()
	}
	if tags["tag_int"] != "42" {
		t.Fail()
	}
	if tags["tag_float"] != "4.25" {
		t.Fail()
	}
}

func _histogram_test_helper(t *testing.T, parsed *StatsdMessage) {
	if parsed.Type != STATSD_TYPE_HISTOGRAM {
		t.Fail()
	}
	if parsed.Name != "tm_1" {
		t.Fail()
	}
	if parsed.Value != 930 {
		t.Fail()
	}
}

func _map_test_helper(t *testing.T, expected map[string]string, to_test map[string]string) {
	// verify that sizes match
	if len(expected) != len(to_test) {
		t.Logf("Maps have different number of keys")
		t.Logf("A: %v\nB: %v\n", expected, to_test)
		t.Fail()
		return
	}
	for key_, val_ := range expected {
		t_val, ok := to_test[key_]
		if !ok {
			t.Logf("Expected key '%s' not found in tested map", key_)
			t.Fail()
			return
		}
		if val_ != t_val {
			t.Logf("Value of '%s' differs: '%s', '%s'", key_, val_, t_val)
			t.Fail()
			return
		}
	}
}

func TestBasicStatsdMessage(t *testing.T) {
	s := "tm_1:930|h"
	sm := ParseStatsdMessage(s)
	if sm == nil {
		t.Logf("Basic message failed to parse: %s", s)
		t.Fail()
		return
	}
	_histogram_test_helper(t, sm)
}

func TestLegacyStatsdMessage(t *testing.T) {
	s := "tm_1:930|h|@15"
	sm := ParseStatsdMessage(s)
	if sm == nil {
		t.Logf("Legacy message failed to parse: %s", s)
		t.Fail()
		return
	}
	_histogram_test_helper(t, sm)
}

func TestMsgWithAllFields(t *testing.T) {
	s := "tm_1:930|h|@15|#foo:bar,blahblah:14"
	sm := ParseStatsdMessage(s)
	if sm == nil {
		t.Logf("All-fields message failed to parse: %s", s)
		t.Fail()
		return
	}
	_histogram_test_helper(t, sm)
	e := map[string]string{"foo": "bar", "blahblah": "14"}
	_map_test_helper(t, e, sm.Tags)
}

// This is the message format we'd most likely expect to see in live use
func TestWithAllButRateFields(t *testing.T) {
	s := "tm_1:930|h|#foo:bar,blahblah:14"
	sm := ParseStatsdMessage(s)
	if sm == nil {
		t.Logf("All-fields message failed to parse: %s", s)
		t.Fail()
		return
	}
	_histogram_test_helper(t, sm)
	e := map[string]string{"foo": "bar", "blahblah": "14"}
	_map_test_helper(t, e, sm.Tags)
}

func TestParsedFields(t *testing.T) {
	s := "ctr_X:1|c|#some:info,in:tags,and:other,random:garbage"
	m, err := ParseToFields(s)
	if err != nil {
		t.Logf("Failed to parse entirely: %s", s)
		t.Fail()
		return
	}
	// For debugging until things clear
	t.Logf("Parsed map: %v", m)

	if m["name"] != "ctr_X" {
		t.Logf("Incorrect name: %s != ctr_X", m["name"])
		t.Fail()
	}
	if m["value"] != "1" {
		t.Logf("Incorrect value: %s != 1", m["value"])
		t.Fail()
	}
	if m["type_char"] != "c" {
		t.Logf("Incorrect type character: %s != c", m["type_char"])
		t.Fail()
	}
	if m["tags"] != "some:info,in:tags,and:other,random:garbage" {
		t.Logf("Incorrect tags from message: %s", m["tags"])
		t.Fail()
	}
}

// Make sure the message used in blaster is parsed correctly
func TestWithCounterAndTags(t *testing.T) {
	s := "ctr_X:1|c|#some:info,in:tags,and:other,random:garbage"
	sm := ParseStatsdMessage(s)
	if sm == nil {
		t.Logf("Message failed to parse: %s", s)
		t.Fail()
		return
	}
	if sm.Name != "ctr_X" {
		t.Logf("Name does not match: '%s' != 'ctr_X'", sm.Name)
		t.Fail()
	}
	if sm.Value != 1.0 {
		t.Logf("Value does not match: '%f' != '1.0'", sm.Value)
		t.Fail()
	}
	if sm.Type != STATSD_TYPE_COUNT {
		t.Logf("Type does not match: '%d' != '0'", sm.Type)
		t.Fail()
	}
	e := map[string]string{"some": "info", "in": "tags", "and": "other", "random": "garbage"}
	_map_test_helper(t, e, sm.Tags)
}
