package main

import (
	"fmt"
	"strconv"
	"strings"
)

// We'll fake an enum
const (
	STATSD_TYPE_COUNT     = 0 // c
	STATSD_TYPE_GAUGE     = 1 // g
	STATSD_TYPE_HISTOGRAM = 2 // h
	// we don't support 'distribution' type
	// we don't support 'rate' type either
)

type StatsdMessage struct {
	Name       string
	Value      float64
	Type       uint8
	SampleRate int // keep this static -1 to prevent anyone from actually using it
	Tags       map[string]string
}

func atof(val string) float64 {
	fv, err := strconv.ParseFloat(val, 64)
	if err != nil {
		// An obviously wrong but still a valid value
		return -1.0
	}
	return fv
}

// Exported to make testable
func ParseTags(tags string) map[string]string {
	out := make(map[string]string)

	pairs := strings.Split(tags, ",")
	for _, item := range pairs {
		if len(item) < 3 {
			continue
		}
		kv := strings.SplitN(item, ":", 2)
		out[kv[0]] = kv[1]
	}

	return out
}

func parse_type(t string) uint8 {
	if t == "c" {
		return STATSD_TYPE_COUNT
	} else if t == "g" {
		return STATSD_TYPE_GAUGE
	} else { // h
		return STATSD_TYPE_HISTOGRAM
	}
}

// This is notably faster than regex-based field extraction.
// Maximum rate with regex-based matcher:   ~280k PPS
// Maximum rate with this string-splitter:  ~410k PPS
//
// The format of DogStatsD messages is:
// <METRIC_NAME>:<VALUE>|<TYPE>|@<SAMPLE_RATE>|#<TAG_KEY_1>:<TAG_VALUE_1>,<TAG_2>
func parse_msg_as_fields(msg_line string) (map[string]string, error) {
	out := make(map[string]string)
	out["name"] = ""
	out["value"] = ""
	out["type_char"] = ""

	// If we split on '#' first, that allows to get tags cheaply
	elems := strings.Split(msg_line, "#")
	if len(elems) > 2 {
		return nil, fmt.Errorf("Invalid message format - too many octothorpes (#)")
	}

	// "No tags" should be a rare case, most people want them
	if len(elems) < 2 {
		out["tags"] = ""
	} else {
		out["tags"] = elems[len(elems)-1]
	}

	// Since we ignore sample rate, we only care about the first two
	// fields - separated neatly by '|'
	stats := elems[0]
	toks := strings.SplitN(stats, "|", 3) // NOTE: different semantics than in Python!

	out["type_char"] = toks[1]

	toks = strings.Split(toks[0], ":")
	if len(toks) != 2 {
		return nil, fmt.Errorf("Invalid message format - no NAME:VALUE")
	}
	out["name"] = toks[0]
	out["value"] = toks[1]

	return out, nil
}

// Exported to make field parser directly testable
func ParseToFields(msg_line string) (map[string]string, error) {
	return parse_msg_as_fields(msg_line)
}

func ParseStatsdMessage(msg_line string) *StatsdMessage {
	m := new(StatsdMessage)
	m.SampleRate = -1

	toks, err := parse_msg_as_fields(msg_line)
	// We duplicate the error label. Jumping over a variable declaration
	// makes the Go compiler unhappy.
	if err != nil {
		WarnAndSuppress(msg_line)
		return nil
	}

	m.Name = toks["name"]
	m.Value = atof(toks["value"])
	m.Type = parse_type(toks["type_char"])
	t, ok := toks["tags"]
	if ok {
		m.Tags = ParseTags(t)
	} else {
		goto Err
	}
	return m

Err:
	WarnAndSuppress(msg_line)
	return nil
}
