package main

import (
	"fmt"
	"net"
	"testing"
	"time"
)

type TestClient struct {
	conn *net.UDPConn
}

var receiver *StatsdListener
var sender *TestClient

func _setup() {
	receiver = NewListener(60235)

	remote, err := net.ResolveUDPAddr("udp", "127.0.0.1:60235")
	if err != nil {
		panic("Failed to resolve server address")
	}
	sender = new(TestClient)
	sender.conn, err = net.DialUDP("udp", nil, remote)
	if err != nil {
		panic("Failed to set up UDP client socket")
	}
	receiver.Start()
}

func _clean() {
	sender.conn.Close()
	receiver.Stop()
	time.Sleep(250 * time.Millisecond)
	receiver.Stats()
	// Dump final state
	for k, val := range receiver.collector.data.Counters {
		fmt.Printf("Counter: %#v --> %v\n", k, val)
	}
	for k, val := range receiver.collector.data.Gauges {
		fmt.Printf("Gauge: %#v --> %v\n", k, val.Tags)
	}
	for k, val := range receiver.collector.data.Histograms {
		fmt.Printf("Histogram: %#v --> %v, %v\n", k, val.Tags, val.Histogram())
	}
	// Rendered page
	fmt.Printf("%s", receiver.collector.RenderPage(receiver.collector.data))
}

func TestMain(m *testing.M) {
	_setup()
	m.Run()
	_clean()
}

func TestSendOne(t *testing.T) {
	m := []byte("tm_1:930|h|@15|#foo:bar,blahblah:14\n")
	_, err := sender.conn.Write(m)
	if err != nil {
		t.Fail()
		panic(fmt.Errorf("Failed to send packet: %v", err))
	}
}

func TestSend4k(t *testing.T) {
	msgs := make(map[string][]byte)
	msgs["a"] = []byte("tm_1:930|h|@15|#foo:bar,blahblah:14\n")
	msgs["b"] = []byte("g_1:19|g|#some1:else,harm:done,evil:1\n")
	msgs["c"] = []byte("c_2:1|c|#nobody:else\n")
	msgs["d"] = []byte("tm_2:11.3|h|#unit:ms\n" +
		"g_1:19|g|#some1:else,harm:done,evil:1\n" +
		"")
	msgs["e"] = []byte("g_2:85|g|#some1:there,harm:undone,evil:bit\n")
	msgs["f"] = []byte("c_7:3|c|#nobody:here\n")
	msgs["g"] = []byte("tm_6:4219|h|@15|#foo:bar,blahblah:17\n")
	for i := 0; i < 500; i++ {
		for _, m := range msgs {
			_, err := sender.conn.Write(m)
			if err != nil {
				t.Fail()
				panic(fmt.Errorf("Failed to send one of many packets (i=%d): %v", i, err))
			}
		}
		if i%20 == 0 {
			time.Sleep(50 * time.Microsecond)
		}
	}
}
