package main

import (
	"fmt"
	"sort"
	"strings"
)

type BucketedCounter struct {
	counter  map[uint64]uint64
	max      uint64
	total    uint64
	num_keys uint64
	Tags     map[string]string
}

func NewBucketedCounter() *BucketedCounter {
	bc := new(BucketedCounter)
	bc.counter = make(map[uint64]uint64)
	bc.Tags = make(map[string]string)
	return bc
}

func (c *BucketedCounter) Total() uint64 {
	return c.total
}

func (c *BucketedCounter) Counts() map[uint64]uint64 {
	return c.counter
}

func (c *BucketedCounter) NumKeys() uint64 {
	return c.num_keys
}

func (c *BucketedCounter) PrometheusTags() string {
	return GetPrometheusTags(c.Tags)
}

func (c *BucketedCounter) HistogramTags() map[string]string {
	provided_tags := GetRawStringTags(c.Tags)
	out := make(map[string]string)

	h := c.Histogram()
	for quantile, _ := range h {
		tag_str := fmt.Sprintf("quantile=\"%s\"", quantile)

		// We don't want to modify the originals by accident, ever
		qt_tags := make([]string, len(provided_tags))
		copy(qt_tags, provided_tags)
		qt_tags = append(qt_tags, tag_str)

		fin := fmt.Sprintf("{%s}", strings.Join(qt_tags, ","))

		out[quantile] = fin
	}
	return out
}

func (c *BucketedCounter) Include(value uint64) {
	_, seen := c.counter[value]
	if !seen {
		c.counter[value] = 0
		c.num_keys += 1
	} else {
		c.counter[value] += 1
	}
	c.total += 1
	if value > c.max {
		c.max = value
	}
}

func (ctr *BucketedCounter) Histogram() map[string]int {
	// Grab copies of keys (usec times) and sort them
	time_buckets := make([]int, ctr.NumKeys())
	i := 0
	for usec := range ctr.counter {
		time_buckets[i] = int(usec)
		i += 1
	}
	sort.Ints(time_buckets)

	// We always create p50, p95, p99, p100 (max)
	hg := make(map[string]int)

	hg["min"] = time_buckets[0]
	hg["max"] = time_buckets[len(time_buckets)-1]
	hg["p100"] = time_buckets[len(time_buckets)-1]
	hg["p99"] = 0
	hg["p95"] = 0
	hg["p50"] = 0

	var elems int = 0

	for j := len(time_buckets) - 1; j > 0; j-- {
		k := time_buckets[j]
		elems += int(ctr.counter[uint64(k)])
		total := int(ctr.Total())

		frac := float32(elems) / float32(total)

		if frac <= 0.01 {
			hg["p99"] = k
			continue
		}
		if frac <= 0.05 {
			hg["p95"] = k
			continue
		}
		if frac <= 0.5 {
			hg["p50"] = k
			continue
		}

		// If we've gone past the halfway mark, we can break. Saves
		// approximately half the runtime
		if frac > 0.5 {
			break
		}
	}

	return hg
}
