package main

import (
	"bytes"
	"fmt"
	"log"
	"net/http"
	"net/http/pprof"
	"sync"
	"text/template"
)

var LOG *log.Logger

// XXX: This works, but the syntax for accessing a map by a known key
// and not just by range iteration is pretty ugly. Sure, I can assign
// the map to a template variable, but then printing out the content of
// m[key_name] is done through the "index m key_name" hoop jump.
var _template string = `####
## INFO counters
{{- range $name, $counter := .Counters }}
# TYPE {{ $name }} counter
{{ $name }}{{ $counter.PrometheusTags }} {{ $counter.Value }}
{{- end }}
## INFO gauges
{{- range $name, $gauge := .Gauges }}
# TYPE {{ $name }} gauge
{{ $name }}{{ $gauge.PrometheusTags }} {{ $gauge.Value }}
{{ end }}
## INFO histograms
{{- range $name, $elem := .Histograms }}
{{ $hist := $elem.Histogram -}}
# TYPE {{ $name }} histogram
{{ range $qkey, $hist_tag := $elem.HistogramTags -}}
{{- $name }}{{ $hist_tag }} {{ index $hist $qkey }}
{{ end -}}
{{ end -}}
`

type Counter struct {
	KeyName string
	Value   uint64
	Tags    map[string]string
}

type Gauge struct {
	KeyName string
	Value   uint64
	Tags    map[string]string
}

// XXX: Identical to gauge. Maybe not needed as separate type?
type HistogramData struct {
	KeyName string
	Value   uint64
	Tags    map[string]string
}

func (c *Counter) PrometheusTags() string {
	return GetPrometheusTags(c.Tags)
}

func (g *Gauge) PrometheusTags() string {
	return GetPrometheusTags(g.Tags)
}

type Aggregations struct {
	Counters   map[string]*Counter
	Gauges     map[string]*Gauge
	Histograms map[string]*BucketedCounter
}

type StatsCollector struct {
	data *Aggregations

	mux                *http.ServeMux
	telemetry_page_tpl *template.Template
	telemetry_page_txt string

	mutex sync.Mutex
}

func init() {
	LOG = log.Default()
}

func NewAggregations() *Aggregations {
	agg := new(Aggregations)

	agg.Counters = make(map[string]*Counter)
	agg.Gauges = make(map[string]*Gauge)
	agg.Histograms = make(map[string]*BucketedCounter)

	return agg
}

func NewStatsCollector() *StatsCollector {
	sc := new(StatsCollector)

	sc.data = NewAggregations()

	sc.telemetry_page_tpl = template.Must(template.New("prom_page").Parse(_template))

	sc.mux = http.NewServeMux()
	sc.mux.Handle("/metrics", sc.TelemetryPageHandler())

	// Profiler. Lifted from https://stackoverflow.com/questions/35360089/
	sc.mux.HandleFunc("/debug/pprof/", pprof.Index)
	sc.mux.HandleFunc("/debug/pprof/{action}", pprof.Index)
	sc.mux.HandleFunc("/debug/pprof/symbol", pprof.Symbol)
	sc.mux.HandleFunc("/debug/pprof/profile", pprof.Profile)

	sc.telemetry_page_txt = "This page intentionally left blank\n"

	return sc
}

func (sc *StatsCollector) Start(port uint16) {
	port_definition := fmt.Sprintf(":%d", port)
	log.Fatal(http.ListenAndServe(port_definition, sc.mux))
}

func (sc *StatsCollector) Flip(verbose bool) {
	if verbose {
		LOG.Println("Flipping over to new data collector")
	}

	// Do the structure allocations before the render
	next_agg := NewAggregations()

	sc.mutex.Lock()
	old_agg := sc.data
	sc.data = next_agg
	sc.mutex.Unlock()

	sc.telemetry_page_txt = sc.RenderPage(old_agg)
}

// Two layers of indirection, since the handler is thrown into its own
// goroutine.
func (sc *StatsCollector) TelemetryPageHandler() http.Handler {
	return http.HandlerFunc(sc.write_page)
}

func (sc *StatsCollector) write_page(w http.ResponseWriter, r *http.Request) {
	w.Header().Set("Content-Type", "text/text; charset=utf-8")
	w.WriteHeader(http.StatusOK)
	bytes, err := w.Write([]byte(sc.telemetry_page_txt))
	if err != nil {
		LOG.Printf("Connection error after sending %dB: %v", bytes, err)
		return
	}
}

func (sc *StatsCollector) RenderPage(agg_data *Aggregations) string {
	var b bytes.Buffer
	err := sc.telemetry_page_tpl.Execute(&b, agg_data)
	if err != nil {
		return fmt.Sprintf("This page unintentionally left blank\n%v", err)
	}
	return b.String()
}
